/*
 *  Copyright (c) 2000-2003 Barak Weichselbaum <barak@komodia.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Contact info:
 * -------------
 *
 * Site:					http://www.komodia.com
 * Main contact:			barak@komodia.com
 * For custom projects, 
 * consulting, or other
 * paid services:			sales@komodia.com
 */

#include "stdafx.h"
#include "UDPRelay.h"

#include "ErrorHandlerMacros.h"
#include "GenericCriticalSection.h"
#include "OSManager.h"

#ifdef _MEMORY_DEBUG 
	#define new	   DEBUG_NEW  
	#define malloc DEBUG_MALLOC  
    static char THIS_FILE[] = __FILE__;  
#endif

KOMODIA_NAMESPACE_START

// ----------------------------- CListenSocket start ---------------------

#define CListenSocket_Class "CListenSocket"

CUDPRelay::CListenSocket::CListenSocket(CUDPRelay* pFather) : CUDPSocketAsync(FALSE),
															  m_pFather(pFather)
{
	try
	{
		//Set our name
		SetName(CListenSocket_Class);
	}
	ERROR_HANDLER("CListenSocket")
}

CUDPRelay::CListenSocket::~CListenSocket()
{
}

BOOL CUDPRelay::CListenSocket::OnSocketReceive(int iErrorCode)
{
	try
	{
		if (iErrorCode)
		{
			//Report it
			ReportError("OnSocketReceive","Received an error code!",iErrorCode);

			//Done
			return FALSE;
		}

		char cBuffer[65536];

		//Data information
		IP aIP;
		unsigned short usPort;

		//Get the data
		int iResult;
		iResult=Receive(cBuffer,
					    sizeof(cBuffer),
						aIP,
						usPort);

		//Did we receive anything?
		if (iResult>0)
			//Send the data
			m_pFather->SendData(cBuffer,
								iResult,
								aIP,
								usPort);
		else
			//Report the error
			ReportErrorOS("OnSocketReceive","Failed to receive data!");

		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("OnSocketReceive",FALSE)
}

// ----------------------------- CListenSocket end ---------------------

// ----------------------------- CClientSocket start ---------------------

#define CClientSocket_Class "CClientSocket"

CUDPRelay::CClientSocket::CClientSocket(CUDPRelay* pFather,
										IP aSourceAddress,
										unsigned short usSourcePort) : CUDPSocketAsync(FALSE),
																	   m_pFather(pFather),
																	   m_aSourceAddress(aSourceAddress),
																	   m_usSourcePort(usSourcePort),
																	   m_usLocalPort(0)
{
	try
	{
		//Set our name
		SetName(CClientSocket_Class);

		//Get ourselves a port
		m_usLocalPort=m_pFather->AllocateLocalPort();
	}
	ERROR_HANDLER("CClientSocket")
}

CUDPRelay::CClientSocket::~CClientSocket()
{
}

BOOL CUDPRelay::CClientSocket::OnSocketReceive(int iErrorCode)
{
	try
	{
		if (iErrorCode)
		{
			//Report it
			ReportError("OnSocketReceive","Received an error code!",iErrorCode);

			//Done
			return FALSE;
		}

		//Do we have a timer
		if (m_pFather->GetTimeout())
			//Set it
			if (!SetTimeout(m_pFather->GetTimeout()))
				//Report it
				ReportError("OnSocketReceive","Failed to set timeout!");

		char cBuffer[65536];

		//Data information
		IP aIP;
		unsigned short usPort;

		//Get the data
		int iResult;
		iResult=Receive(cBuffer,
					    sizeof(cBuffer),
						aIP,
						usPort);

		//Did we receive anything?
		if (iResult>0)
		{
			//Get the socket
			CListenSocket* pSocket;
			pSocket=m_pFather->m_pSocket;

			//Send the data
			if (pSocket &&
				pSocket->Send(m_aSourceAddress,
							  m_usSourcePort,
							  cBuffer,
							  iResult)<=0)
				//Report the error
				ReportError("OnSocketReceive","Failed to send data!");
			else
				return TRUE;
		}
		else
			//Report the error
			ReportErrorOS("OnSocketReceive","Failed to receive data!");

		//Done
		return FALSE;
	}
	ERROR_HANDLER_RETURN("OnSocketReceive",FALSE)
}

int CUDPRelay::CClientSocket::SendData(const char* pBuffer,
									   unsigned long ulBufferSize)
{
	try
	{
		//Do we have a timer
		if (m_pFather->GetTimeout())
			//Set it
			if (!SetTimeout(m_pFather->GetTimeout()))
				//Report it
				ReportError("SendData","Failed to set timeout!");

		//Send the data
		return Send(m_pFather->GetTarget(),
					m_pFather->GetTargetPort(),
					pBuffer,
					ulBufferSize);
	}
	ERROR_HANDLER_RETURN("SendData",GetErrorCode())
}

BOOL CUDPRelay::CClientSocket::OnSocketTimeout()
{
	try
	{
		//Remove ourselves
		m_pFather->RemoveSocket(m_aSourceAddress,
								m_usSourcePort);

		//Delete ourselves
		delete this;

		//Done
		return FALSE;
	}
	ERROR_HANDLER_RETURN("OnSocketTimeout",FALSE)
}

// ----------------------------- CClientSocket end ---------------------

#define CUDPRelay_Class "CUDPRelay"

CUDPRelay::CUDPRelay() : CRelay(),
						 m_pSocket(NULL),
						 m_bCreated(FALSE),
						 m_aTarget(0),
						 m_usTargetPort(0),
						 m_pCSection(NULL),
						 m_pCSectionPorts(NULL),
						 m_usBindPort(0)
{
	try
	{
		//Set our name
		SetName(CUDPRelay_Class);

		//Create the CS
		m_pCSection=COSManager::CreateCriticalSection();
		m_pCSectionPorts=COSManager::CreateCriticalSection();
	}
	ERROR_HANDLER("CUDPRelay")
}

CUDPRelay::~CUDPRelay()
{
	try
	{
		delete m_pCSection;
		delete m_pCSectionPorts;
	}
	ERROR_HANDLER("~CUDPRelay")
}

BOOL CUDPRelay::Relay(IP aBindAddress,
					  unsigned short usBindPort,
					  IP aDestinationAddress,
					  unsigned short usDestinationPort)
{
	try
	{
		//Are we created?
		if (m_bCreated)
		{
			//Report it
			ReportError("Relay","Already created!");

			//Exit
			return FALSE;
		}

		//Create the socket
		m_pSocket=new CListenSocket(this);

		//Protect it
		std::auto_ptr<CListenSocket> pProtection(m_pSocket);

		//Try to create it
		if (!m_pSocket->Create())
		{
			//Report it
			ReportError("Relay","Failed to create socket!");

			//Exit
			return FALSE;
		}

		//Try to bind it
		if (!m_pSocket->Bind(aBindAddress,
							 usBindPort))
		{
			//Report it
			ReportError("Relay","Failed to create socket!");

			//Exit
			return FALSE;
		}

		//Save the data
		m_aTarget=aDestinationAddress;
		m_usTargetPort=usDestinationPort;
		m_usBindPort=usBindPort;

		//Set local port
		m_usLocalPort=1024;

		//Try to listen
		if (!m_pSocket->Listen())
		{
			//Report it
			ReportError("Relay","Failed to listen!");

			//Exit
			return FALSE;
		}

		//Remove protection
		pProtection.release();

		//We are done
		m_bCreated=TRUE;

		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("Relay",FALSE)
}

BOOL CUDPRelay::Relay(const std::string& rBindAddress,
					  unsigned short usBindPort,
					  const std::string& rDestinationAddress,
					  unsigned short usDestinationPort)
{
	try
	{
		return Relay(CSocketBase::StringToLong(rBindAddress),
					 usBindPort,
					 CSocketBase::StringToLong(rDestinationAddress),
					 usDestinationPort);
	}
	ERROR_HANDLER_RETURN("Relay",FALSE)
}

IP CUDPRelay::GetTarget()const
{
	return m_aTarget;
}

unsigned short CUDPRelay::GetTargetPort()const
{
	return m_usTargetPort;
}

unsigned short CUDPRelay::GetBindPort()const
{
	return m_usBindPort;
}

void CUDPRelay::SendData(const char* pBuffer,
						 unsigned long ulBufferSize,
						 IP aDestinationIP,
						 unsigned short usDestinationPort)
{
	try
	{
		//Search for the socket
		ConnectionData aData;
		aData.aSourceIP=aDestinationIP;
		aData.usSourcePort=usDestinationPort;

		//The socket
		CClientSocket* pSocket;
		pSocket=NULL;

		{
			//Lock the data
			CCriticalAutoRelease aRelease(m_pCSection);
			
			//Do we have it
			SocketMap::iterator aIterator;
			aIterator=m_aSocketMap.find(aData);
			if (aIterator!=m_aSocketMap.end())
				//Save the socket
				pSocket=aIterator->second;
		}

		//Do we have the socket
		if (pSocket)
			//Send the data
			if (pSocket->SendData(pBuffer,
								  ulBufferSize)<=0)
				//Report it
				ReportErrorOS("SendData","Failed to send data!");
			else
				;
		else
		{
			//Create the socket
			if (!(pSocket=CreateSocket(aDestinationIP,
									   usDestinationPort)))
				//Report it
				ReportError("CreateSocket","Failed to create socket!");
			else
				//Recall ourselves
				SendData(pBuffer,
						 ulBufferSize,
						 aDestinationIP,
						 usDestinationPort);
		}
	}
	ERROR_HANDLER("SendData")
}

unsigned short CUDPRelay::AllocateLocalPort()const
{
	try
	{
		//Lock
		CCriticalAutoRelease aRelease(m_pCSectionPorts);

		//Return the port
		return ++m_usLocalPort;
	}
	ERROR_HANDLER_RETURN("AllocateLocalPort",m_usLocalPort)
}

void CUDPRelay::RemoveSocket(IP aSourceAddress,
							 unsigned short usSourcePort)
{
	try
	{
		//Add the socket
		ConnectionData aData;
		aData.aSourceIP=aSourceAddress;
		aData.usSourcePort=usSourcePort;

		//Lock
		CCriticalAutoRelease aRelease(m_pCSection);

		//Insert the socket
		m_aSocketMap.erase(aData);
	}
	ERROR_HANDLER("RemoveSocket")
}

CUDPRelay::CClientSocket* CUDPRelay::CreateSocket(IP aSourceAddress,
												  unsigned short usSourcePort)
{
	try
	{
		//Create the socket
		CClientSocket* pSocket;
		pSocket=new CClientSocket(this,
								  aSourceAddress,
								  usSourcePort);

		//Protect the socket
		std::auto_ptr<CClientSocket> pProtection(pSocket);

		//Try to create it
		if (!pSocket->Create())
		{
			//Report it
			ReportError("CreateSocket","Failed to create socket!");

			//Exit
			return NULL;
		}
		
		//Try to create it
		if (!pSocket->Listen())
		{
			//Report it
			ReportError("CreateSocket","Failed to create socket!");

			//Exit
			return NULL;
		}

		//Add the socket
		ConnectionData aData;
		aData.aSourceIP=aSourceAddress;
		aData.usSourcePort=usSourcePort;

		{
			//Lock
			CCriticalAutoRelease aRelease(m_pCSection);

			//Insert the socket
			m_aSocketMap.insert(SocketMap::value_type(aData,pSocket));
		}

		//Release the protection
		pProtection.release();

		//Done
		return pSocket;
	}
	ERROR_HANDLER_RETURN("CreateSocket",NULL)
}

BOOL CUDPRelay::Stop()
{
	try
	{
		//Are we created?
		if (!m_bCreated)
			return TRUE;

		//Delete the listener
		m_pSocket->DeleteSocketFromThread();
		m_pSocket=NULL;

		{
			//Lock
			CCriticalAutoRelease aRelease(m_pCSection);

			//Iterate the data
			SocketMap::iterator aIterator;
			aIterator=m_aSocketMap.begin();
			while (aIterator!=m_aSocketMap.end())
			{
				//Signal the socket
				aIterator->second->DeleteSocketFromThread();

				//Erase
				aIterator=m_aSocketMap.erase(aIterator);
			}
		}

		//We are not created
		m_bCreated=FALSE;

		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("Stop",FALSE)
}

KOMODIA_NAMESPACE_END
